function [ExpData,NUSData] = Sampling()

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% INPUT VALUES %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Experimental parameters
sample = 'SH3';         % Sample's name, tag or label
SF = 700;               % Spectrometer frequency in MHz
sz1 = 256;              % Data points in indirect dimension
sz2 = 200;              % Data points in direct dimension
SW1 = 2500.00;          % Spectral Width in indirect dimension
SW2 = 2239.822;         % Spectral Width in direct dimension
zff1 = 4;               % Zero-filling factor in dimension 1 (indirect)
zff2 = 4;               % Zero-filling factor in dimension 2 (direct)
files = 0:8;            % Files for processing
G0 = 21.12e-6;          % Initial protein concentration [mol/L]
% Concentrations (ligand:protein)
R = [2.07 9.3 28.93 49.59 76.45 113.64 165.29 206.61];
% Dilution compensation factors
D = [1.0018 1.0082 1.0255 1.0436 1.0673 1.1 1.1455 1.1818];
G = G0./D;

% Coordinates of maxima of mother spectrum for selected peaks
% At the end of this function there is a script that
% can be used to plot spectra and find these values.
%       D1  D2
cenpts=[593 302;
        651 596;
        400 188;
        778 125;
        615 323;
        589 503;
        330 186;
        320 219;
        489 127;
        380 180;
        443 350;
        769 775;
        532 436];
% Zero filling factors used for recording cenpts values
        zff1cp = 4;
        zff2cp = 4;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% PRE-PROCESS ------------------------------------------------------------%
SF1 = (sz1/(sz1-1))*SW1;    % Sampling frequency t1
SF2 = (sz2/(sz2-1))*SW2;    % Sampling frequency t2

% Adjust center coordinates of peaks according to zero filling factors.
cenpts = round([cenpts(:,1)*(zff1/zff1cp) cenpts(:,2)*(zff2/zff2cp)]);
ws1 = round(1.2*zff1);      % windows size 1
ws2 = round(1.2*zff2);      % windows size 2

% Create window location of peaks
winloc(:,1) = cenpts(:,1)-repmat(ws1,size(cenpts,1),1);
winloc(:,2) = cenpts(:,1)+repmat(ws1,size(cenpts,1),1);
winloc(:,3) = cenpts(:,2)-repmat(ws2,size(cenpts,1),1);
winloc(:,4) = cenpts(:,2)+repmat(ws2,size(cenpts,1),1);
% VALIDATION -------------------------------------------------------------%

% Validate experimental data
validateattributes(sz1,{'numeric'},{'>',0,'real','scalar'});
validateattributes(sz2,{'numeric'},{'>',0,'real','scalar'});
validateattributes(SW1,{'numeric'},{'>',0,'real'});
validateattributes(SW2,{'numeric'},{'>',0,'real'});
validateattributes(zff1,{'numeric'},{'>=',1,'real','scalar'});
validateattributes(zff2,{'numeric'},{'>=',1,'real','scalar'});
validateattributes(files,{'numeric'},{'row','>=',0,'increasing'});
validateattributes(G,{'numeric'},{'row','>',0,'decreasing'});
validateattributes(R,{'numeric'},{'row','>',0,'increasing'});
validateattributes(D,{'numeric'},{'row','>',1,'increasing'});
% SAMPLING ---------------------------------------------------------------%

% Basic definitions and preallocation
Nfiles = length(files);
fid = zeros(2*sz1*zff1,2*sz2*zff2,Nfiles);
nuslist = zeros(sz1/(Nfiles-1),Nfiles-1);
R_scheme = zeros(sz1,1);
G_scheme = zeros(sz1,1);

for i = 1:Nfiles
    file = fopen([int2str(files(i)),'.fid']);
    aux = fread(file,'single');
    if(i == 1)
        fid(:,:,i) = reshape(aux(513:end),2*sz1*zff1,2*sz2*zff2);
    else
        fid(:,:,i) = reshape(aux(513:end),2*sz1*zff1,2*sz2*zff2).*D(i-1);
        nuslist(:,i-1) = dlmread(['nus',int2str(files(i)),'.txt'])+1;
        R_scheme(nuslist(:,i-1),1) = R(i-1);
        G_scheme(nuslist(:,i-1),1) = G(i-1);
    end
    fclose(file);
end

data = fid(1:end/2,:,:)+1i*fid(end/2+1:end,:,:);
data(:,:,2) = sum(data(:,:,2:end),3);
fidmo = real(fftshift(fft(data,sz1*zff1,1),1));
fidmocplx = fidmo(:,1:2:end,:)+1i*fidmo(:,2:2:end,:);
spectra = real(fftshift(fft(fidmocplx,sz2*zff2,2),2));
NUS_signal = squeeze(data(1:sz1,:,2));

% Estimation of CIS values boundaries
amps = dlmread('CIS_boundaries.txt');

% Cell arrays with data
ExpData = {sample,SF,[sz1 sz2],[zff1 zff2],[SF1 SF2],G,R,cenpts,winloc};
NUSData = {NUS_signal,R_scheme,G_scheme,spectra,amps};
%%%%%%%%%%%%%%%%%%%%%%%%%%% PLOTTING SCRIPTS %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% PLOT NUS AND MOTHER SPECTRA --------------------------------------------%
Nfiles = length(files);
fid = zeros(2*sz1*zff1,2*sz2*zff2,Nfiles);
nuslist = zeros(sz1/(Nfiles-1),Nfiles-1);

for i = 1:Nfiles
    file = fopen([int2str(files(i)),'.fid']);
    aux = fread(file,'single');
    if(i == 1)
        fid(:,:,i) = reshape(aux(513:end),2*sz1*zff1,2*sz2*zff2);
    else
        fid(:,:,i) = reshape(aux(513:end),2*sz1*zff1,2*sz2*zff2).*D(i-1);
        nuslist(:,i-1) = dlmread(['nus',int2str(files(i)),'.txt'])+1;
    end
    fclose(file);
end

data = fid(1:end/2,:,:)+1i*fid(end/2+1:end,:,:);
data(:,:,2) = sum(data(:,:,2:end),3);
fidmo = real(fftshift(fft(data,sz1*zff1,1),1));
fidmocplx = fidmo(:,1:2:end,:)+1i*fidmo(:,2:2:end,:);
spectra = real(fftshift(fft(fidmocplx,sz2*zff2,2),2));

M = max(max(spectra(:,:,1)));
v = [0.8 0.6 0.4 0.2 0.1]*M;
figure()
contour(spectra(:,:,1),v,'k','LineWidth',2),hold on
contour(spectra(:,:,2),v,'r','LineWidth',2)

file = fopen('labels.txt');
L = textscan(file,'%s');
L = L{1};
fclose(file);

for i = 1:size(winloc,1)
    rectangle('Position',[winloc(i,3) winloc(i,1)...
        winloc(i,4)-winloc(i,3) winloc(i,2)-winloc(i,1)],'LineWidth',2);
    text(winloc(i,3),winloc(i,1),L(i),'FontSize',18);
end

end