function res = NOSCO(Sampling,OptParam,varargin)
% Non-Stationary COmplementary NUS spectroscopy method for determination
% of dissociation constants (Kd) in protein-ligand complex using a single
% site binding model.
%
% Usage:
%   NOSCO(Sampling,OptParams)
%   NOSCO(Sampling,OptParams,name,value)
%
% Input:
%   Sampling -- A function, specified using @, that has no arguments and
%               that returns the cell arrays ExpData and NUSData.
%   OptParam -- Cell array containing optimization parameters defined in
%               RunMe script.
%
% Optional input:
%   varargin -- name/value pairs
%   'v0',v0 -- initial velocity
%   'c1',c1 -- pbest acceleration factor
%   'c2',c2 -- gbest acceleration factor
%   'wmax',wmax -- maximum inertia weight
%   'wmin',wmin -- minimum inertia weight
%   'pngres',pngres -- set png resolution (def '-r200')
%   'Kr',[Kmin Kmax] -- range for searching Kd (default: automatically)
%
% Optional output:
%   res -- structure with results
%   res.sample -- sample name
%   res.estimations -- matrix with NOSCO estimation values
%   res.signal -- NUS signal
%   res.Rscheme -- ligand:protein concentrations from NUS schedule
%   res.Gscheme -- total protein concentrations from NUS schedule
%   res.spectra -- mother and NUS spectra
%   res.CISamplitudes -- CIS amplitudes used to estimate CIS values
%   res.Nparticles -- number of particles used in PSO
%   res.maxite -- iterations used in PSO
%   res.maxrun -- run used in PSO
%   res.tolerance -- tolerance provided for PSO
%   res.Krange -- range used to look for K values (in order of magnitude)
%   res.v0 -- initial particles velocity
%   res.c1 -- acceleration factor 1
%   res.c2 -- acceleration factor 2
%   res.wmax -- maximum weighing factor
%   res.wmin -- minimum weighing factor
%-------------------------------------------------------------------------%

    % minimal check
    narginchk(2,16)
    nargoutchk(0,1)
    
    % Default varargin values
    v0 = 0.1;           % Initial velocity
    c1 = 0.4;           % Acceleration factor
    c2 = 1;             % Acceleration factor
    wmax = 0.9;         % Inertia weight
    wmin = 0.3;         % Inertia weight
    pngres = '-r200';   % plot resolution
    Kr = [];            % Kd range
    
    if ~isempty(varargin)
        for n = 1:2:length(varargin)
            switch lower(varargin{n})
                case 'v0'
                    % set particles initial velocity
                    v0 = varargin{n+1};
                    validateattributes(v0,{'numeric'},{'real'});
                case 'c1'
                    % set pbest acceleration factor
                    c1 = varargin{n+1};
                    validateattributes(c1,{'numeric'},{'real'});
                case 'c2'
                    % set gbest acceleration factor
                    c2 = varargin{n+1};
                    validateattributes(c2,{'numeric'},{'real'});
                case 'wmax'
                    % set maximum inertia weight
                    wmax = varargin{n+1};
                    validateattributes(wmax,{'numeric'},{'>=',0,'<=',1});
                case 'wmin'
                    % set minimum inertia weight
                    wmin = varargin{n+1};
                    validateattributes(wmax,{'numeric'},{'>=',0,'<=',wmax});
                case 'pngres'
                    % set png resolution
                    pngres = varargin{n+1};
                case 'kr'
                    % set particles initial velocity
                    Kr = varargin{n+1};
                    validateattributes(Kr,{'numeric'},{'real','increasing'});
                otherwise
                    error('Unknown property name %s',varargin{n})
            end
        end
    end
    
    % Get sampling data
    [ExpData,NUSData] = Sampling();
    
    % Copy experimental data
    sample = ExpData{1};
    SF = ExpData{2};
    sz1 = ExpData{3}(1);
    sz2 = ExpData{3}(2);
    zff1 = ExpData{4}(1);
    zff2 = ExpData{4}(2);
    SF1 = ExpData{5}(1);
    SF2 = ExpData{5}(2);
    G = ExpData{6};
    R = ExpData{7};
    cenpts = ExpData{8};
    winloc = ExpData{9};

    % Copy NUS data
    NUS_signal = NUSData{1};
    R_scheme = NUSData{2};
    G_scheme = NUSData{3};
    spectra = NUSData{4};
    amps = NUSData{5};
    
    % Copy optimization parameters
    N = OptParam{1};
    maxite = OptParam{2};
    maxrun = OptParam{3};
    tol = OptParam{4};
    
    % Validate experimental data
    validateattributes(SF1,{'numeric'},{'>',0});
    validateattributes(SF2,{'numeric'},{'>',0});
    validateattributes(cenpts,{'numeric'},{'>',0,'integer'});
    
    % Validate NUS data
    validateattributes(NUS_signal,{'numeric'},{'finite','nonnan'});
    validateattributes(R_scheme,{'numeric'},{'>=',0});
    validateattributes(G_scheme,{'numeric'},{'>=',0});
    validateattributes(spectra,{'numeric'},{'finite','nonnan'});
    validateattributes(amps,{'numeric'},{'finite','nonnan'});
    
    % Validate optimization data
    validateattributes(N,{'numeric'},{'>=',1,'real','scalar'});
    validateattributes(maxite,{'numeric'},{'>=',1,'real','scalar'});
    validateattributes(maxrun,{'numeric'},{'>=',1,'real','scalar'});
    validateattributes(tol,{'numeric'},{'real','>=',0,'<=',1});
    % MAIN CALCULATIONS --------------------------------------------------%
    
    % Calculate Kd boundaries
    if isempty(Kr)
        fun = @(x) KLB(G,R,x);
        Kr(1) = fzero(fun,0.1);
        Kr(2) = min(Kr(1)+3,-1);
        fprintf('\nRange to search for Kd automatically set to ')
        fprintf('[10^%.2f, 10^%.2f]\n',Kr(1),Kr(2));
    end
    
    % Calibrate CIS boundaries
    HmaxCIS = max(max(abs(amps(:,3:4))));
    NmaxCIS = max(max(abs(amps(:,1:2))));
    amps(:,3:4) = amps(:,3:4)*(0.5*SF/HmaxCIS);
    amps(:,1:2) = amps(:,1:2)*(3*(SF/9.865)/NmaxCIS);

    % Get maximum values of mother peaks
    mom_max = Reference(spectra(:,:,1),winloc);

    % Define time vectors
    t1 = (0:sz1-1)./SF1;
    t2 = (0:sz2*zff2-1)./SF2;
    
    % Pre-allocate output
    num = size(cenpts,1);
    est = zeros(num,4);

    % Correct peaks one by one
    for pn = 1:num
        tic
        est(pn,:) = Correction(amps(pn,:),winloc(pn,:),mom_max(pn),tol);
        toc
    end
    % POST-PROCESSING ----------------------------------------------------%
    
    % Remove outliers
    Kavg = mean(est(:,3));
    Kstd = std(est(:,3));
    good = zeros(num,1);
    for k = 1:num
        if abs(est(k,3)-Kavg)<2*Kstd
            good(k,1) = k;
            est(k,5) = 1;
        else
            est(k,5) = 0;
        end
    end
    good = nonzeros(good);
    num = numel(good);
    % SAVE RESULTS -------------------------------------------------------%

    if nargout > 0
        res.sample = sample;
        res.estimations = est;
        res.signal = NUS_signal;
        res.Rscheme = R_scheme;
        res.Gscheme = G_scheme;
        res.spectra = spectra;
        res.CISamplitudes = amps;
        res.Nparticles = N;
        res.maxite = maxite;
        res.maxrun = maxrun;
        res.tolerance = tol;
        res.Krange = Kr;
        res.v0 = v0;
        res.c1 = c1;
        res.c2 = c2;
        res.wmax = wmax;
        res.wmin = wmin;
    end
    % PLOTTING -----------------------------------------------------------%
    
    params = dlmread('classical_fit.txt');
    
    Kfit = params(1,5);
    err = params(1,6);
    file = fopen('labels.txt');
    L = textscan(file,'%s');
    L = L{1};
    fclose(file);
    
    c1names = {'MarkerFaceColor'};
    c1vals = {[0.3467 0.5360 0.6907]};
    c2names = {'MarkerFaceColor'};
    c2vals = {[0.9153 0.2816 0.2878]};
    c3names = {'MarkerFaceColor','LineWidth'};
    c3vals = {[0.3467 0.5360 0.6907],1.5};
    
    figure()
    subplot(2,2,1)
    scatter(1:num,params(good,1),'o',c1names,c1vals),hold on
    scatter(1:num,est(good,1),'s',c2names,c2vals),title('CIS1')
    xlim([0.75 num+0.25])
    ylim([min(amps(:,1)) max(amps(:,2))])
    ylabel('\delta_{1} [Hz]')
    set(gca,'XTick',1:num,'xticklabel',L(good),'XTickLabelRotation',45)
    box on
    
    subplot(2,2,3)
    scatter(1:num,params(good,2),'o',c1names,c1vals),hold on
    scatter(1:num,est(good,2),'s',c2names,c2vals),title('CIS2')
    xlim([0.75 num+0.25])
    ylim([min(amps(:,3)) max(amps(:,4))])
    ylabel('\delta_{2} [Hz]')
    set(gca,'XTick',1:num,'xticklabel',L(good),'XTickLabelRotation',45)
    box on
    
    subplot(2,2,[2 4])
    errorbar(1:num,params(good,3),params(good,4),'o',c3names,c3vals),hold on
    scatter(1:num,est(good,3),'s',c2names,c2vals),title('K_{d} values')
    h = fill([0.75,num+0.25,num+0.25,0.75],...
        [Kfit-2*err,Kfit-2*err,Kfit+2*err,Kfit+2*err],'g',...
        'FaceAlpha',0.15,'EdgeColor','none');
    [~,lg] = legend('Classical','NOSCO','95% CI classical');
    PatchInLegend = findobj(lg, 'type', 'patch');
    set(PatchInLegend(1), 'FaceAlpha', 0.15);
    h.Annotation.LegendInformation.IconDisplayStyle = 'off';
    line([0.75,num+0.25],[Kfit,Kfit],...
        'Color',c1vals{1},'LineWidth',2)
    line([0.75,num+0.25],[mean(est(good,3)),mean(est(good,3))],...
        'Color',c2vals{1},'LineWidth',2)
    xlim([0.75 num+0.25])
    ylim([10^Kr(1) 10^Kr(2)])
    ylabel('K_{d} [M]')
    set(gca,'yscale','log','XTick',1:num,'xticklabel',L(good),...
        'XTickLabelRotation',45)
    KstrC = sprintf('%.1e',Kfit);
    EstrC = sprintf('%.1e',2*err);
    KstrN = sprintf('%.1e',mean(est(good,3)));
    EstrN = sprintf('%.1e',2*std(est(good,3)));
    strC = ['Classical K_{d} = ',KstrC,'\pm',EstrC];
    strN = ['NOSCO K_{d} = ',KstrN,'\pm',EstrN];
    str = [strC char(10) strN];
    dim = [0.59 0.15 0.1 0.1];
    annotation('textbox',dim,'String',str,'FitBoxToText','on','FontSize',8);
    box on
    
    % Save output figure with auto-renaming
    fname = sprintf('%s_figure',sample);
    file = dir(strcat(fname,'*'));
    count = size({file.name},2);
    fname = strcat(fname,'_(',num2str(count+1),').png');
    print(gcf,fname,'-dpng',pngres);
    
    % Save output structure 'res' with auto-renaming
    str = [sample,'_res*'];
    file = dir(str);
    count = size({file.name},2);
    save([sample,'_res_(',num2str(count+1),').mat'],'res');
    % FUNCTIONS ----------------------------------------------------------%

    % Function to find Kd lower boundary
    function val = KLB(G,R,x)
        P = G.*(1+R)+(10^x);
        Chi = (P-sqrt(P.^2-R.*(2*G).^2))./(2*G);
        val = 100*(Chi(end)-Chi(end-1))/Chi(end-1)-1;
    end
    
    % Read mother peaks maxs
    function mom_max = Reference(momspec,winloc)

        mom_max = zeros(1,size(winloc,1));
        for i = 1:size(winloc,1)
            A = momspec(winloc(i,1):winloc(i,2),winloc(i,3):winloc(i,4));
            mom_max(i) = max(A(:));
        end

    end

    % Main correction function
    function res = Correction(amps,winloc,mom_max,tol)

            function peak_max = local(winloc,mom_max,params)

                P = G_scheme.*(1+R_scheme)+params(3);
                Chi = (P-sqrt(P.^2-R_scheme.*(2*G_scheme).^2))./(2*G_scheme);
                arg_t1 = repmat(Chi.*t1',[1 2*sz2*zff2]);
                arg_t2 = repmat(Chi*t2,[2 1]);

                % This is the main block were the correction is applied (in
                % 2 steps to deal with the hypercomplex arrays)
                corr1 = NUS_signal.*exp(-1i*2*pi*(arg_t1*params(1)));   % Correction step 1
                data = [real(corr1);imag(corr1)];                       % Hypercomplex
                fidmocplx0 = data(:,1:2:end)+1i*data(:,2:2:end);        % Rebuild complex signal
                corr2 = fidmocplx0.*exp(-1i*2*pi*arg_t2*params(2));     % Correction step 2
                fidmo = real(fftshift(fft(corr2,sz2*zff2,2),2));        % FT in D2
                fidmocplx = fidmo(1:end/2,:)+1i*fidmo(end/2+1:end,:);   % Rebuild complex signal
                spectrum = real(fftshift(fft(fidmocplx,sz1*zff1,1),1)); % FT in D1
                peak = spectrum(winloc(1):winloc(2),winloc(3):winloc(4));
                peak_max = abs(mom_max - max(peak(:)));
            end

        % PSO
        func = @(params) local(winloc,mom_max,params);
        lb = [amps(1),amps(3),10^Kr(1)];    % Lower bounds
        ub = [amps(2),amps(4),10^Kr(2)];    % Upper bounds
        tol = tol*mom_max;                  % Tolerance
        
        [res,M] = pso(func,lb,ub,tol);      % Call PSO method
        res(1,4) = mom_max-M;
    end

    % PSO method
    function [rfgbest,fffmin] = pso(fun,LB,UB,tol)

        if (tol == 0)
            fprintf('\nNo tolerance provided!')
        end

        m = numel(LB);
        fff = zeros(1,maxrun);
        rgbest = zeros(maxrun,m);

        % run loop ---------------------------------------------------start
        for run = 1:maxrun
            % pso initialization
            for i = 1:N
                parfor j = 1:m-1
%                     rng('shuffle');
                    x0(i,j) = LB(j)+rand()*(UB(j)-LB(j));
                end
            end
            % logarithmic distribution of initial K values
            Kinit = logspace(log10(LB(m)),log10(UB(m)),N);
            x0(:,m) = Kinit(randperm(N));

            x = x0;     % initial population, n x m
            v = v0*x0;  % initial velocity

            parfor i = 1:N
                f0(i) = fun(x0(i,:));
            end
            [fmin0, index0] = min(f0);
            pbest = x0;             % initial pbest
            gbest = x0(index0,:);   % initial gbest
            ite = 0;
            fprintf('\npeak %d, run %d/%d, iteration %d/%d - K = %.2e',...
                pn,run,maxrun,ite,maxite,gbest(1,3));
            
            % iteration loop -----------------------------------------start
            ite = 1;
            while ite<=maxite && fmin0>tol
                fprintf('\npeak %d, run %d/%d, iteration %d/%d - ',...
                    pn,run,maxrun,ite,maxite);
                
                w = wmax-(wmax-wmin)*ite/maxite; % update inertial weight

                % pso velocity updates
                parfor i = 1:N
                    for j = 1:m
%                         rng('shuffle');
                        v(i,j) = w*v(i,j)+c1*rand()*(pbest(i,j)-x(i,j))+...
                                 c2*rand()*(gbest(1,j)-x(i,j));
                        x(i,j) = x(i,j)+v(i,j);
                    end
                end

                % handling boundary violations
                parfor i = 1:N
                    for j = 1:m
                        if x(i,j)<LB(j)
                            x(i,j) = LB(j);
                        elseif x(i,j)>UB(j)
                            x(i,j) = UB(j);
                        end
                    end
                end
                
                % evaluating fitness
                parfor i = 1:N
                    f(i) = fun(x(i,:));
                end

                % updating pbest and fitness
                parfor i = 1:N
                    if f(i)<f0(i)
                        pbest(i,:) = x(i,:);
                        f0(i) = f(i);
                    end
                end

                [fmin,index] = min(f0); % finding out the best particle

                % updating gbest and best fitness
                if fmin<fmin0
                    gbest = pbest(index,:);
                    fmin0 = fmin;
                end
                fprintf('K = %.2e',gbest(1,3));
                ite = ite+1;
            end
            % iteration loop--------------------------------------------end

            fff(run) = fmin0;
            rgbest(run,:) = gbest;
            if fmin0 <= tol
                fprintf('\nDesired tolerance reached!')
                break
            else
            end
            fprintf('\nBest of run %d: K = %.2e',run,gbest(3));

        end
        % run loop -----------------------------------------------------end
        [fffmin,indexff] = max(fff); % finding out the best run
        rfgbest = rgbest(indexff,:);
        fprintf('\n ------------> Global best for peak %d - K = %.2e \n',pn,rfgbest(3));
    end

end